﻿#if       !defined(INCLUDED_SZ_ARCHIVE_PROPERTY)
#define            INCLUDED_SZ_ARCHIVE_PROPERTY

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_STD_BITSET)
#define            INCLUDED_STD_BITSET
#include  <bitset>
#endif // !defined(INCLUDED_STD_BITSET)

SZ_NS_BEG(szpp)

/// <summary>
/// アーカイブのプロパティを表現するクラス。
/// </summary>
/// <remarks>
/// 厳密に言えば本クラスはアーカイブ全体のプロパティを表すので、アーカイブボリュームのプロパティとは区別するべきである。
/// しかし、SevenZip++ の仕様上、アーカイブボリュームは論理的なアーカイブ全体とほぼ同一視されているため、これらを厳格に区別する実益はあまりない。
/// </remarks>
class SZ_SPEC ArchiveProperty
{
public:
  /// デフォルトコンストラクタ。
  ArchiveProperty();
  /// デストラクタ。
  ~ArchiveProperty();

  /// コピーコンストラクタ。
  ArchiveProperty(const ArchiveProperty &rhs);
  /// 代入演算子。
  ArchiveProperty &operator=(const ArchiveProperty &rhs);

  /// アーカイブコメントがあれば（かつ 7-Zip コーデックがそれを処理できれば）コメント文字列を返すメソッド。
  const szstring &GetArchiveComment() const { return archiveComment; }
  /// アーカイブコメントを設定するメソッド。ただし、7-Zip コーデックがそれを処理できないこともある。
  void SetArchiveComment(const szstring &comment) { archiveComment.assign(comment); }

  /// 圧縮サイズを取得するメソッド。単位はバイト。
  u64 GetCompressedSize() const { return compressedSize; }
  /// 圧縮サイズを設定するメソッド。単位はバイト。
  void SetCompressedSize(const u64 &size) { compressedSize = size; }

  /// 展開（非圧縮）サイズを取得するメソッド。単位はバイト。
  u64 GetUncompressedSize() const { return uncompressedSize; }
  /// 展開（非圧縮）サイズを設定するメソッド。単位はバイト。
  void SetUncompressedSize(const u64 &size) { uncompressedSize = size; }

  /// 格納アイテム数を取得するメソッド。
  u32 GetNumberOfItems() const { return numberOfItems; }
  /// 格納アイテム数を設定するメソッド。
  void SetNumberOfItems(u32 items) { numberOfItems = items; }

  /// ボリューム数を取得するメソッド。
  u32 GetNumberOfVolumes() const { return numberOfVolumes; }
  /// ボリューム数を設定するメソッド。
  void SetNumberOfVolumes(u32 volumes) { numberOfVolumes = volumes; }

  /// マルチボリュームであるか否かを取得するメソッド。
  bool GetIsMultivolume() const { return bits.at(BIT_VOLUME); }
  /// マルチボリュームであるか否かを設定するメソッド。
  void SetIsMultivolume(bool flag) { bits.set(BIT_VOLUME, flag); }

  /// ソリッドアーカイブであるか否かを取得するメソッド。
  bool GetIsSolid() const { return bits.at(BIT_SOLID); }
  /// ソリッドアーカイブであるか否かを設定するメソッド。
  void SetIsSolid(bool flag) { bits.set(BIT_SOLID, flag); }

private:

  enum ArchivePropertyBit
  {
    BIT_VOLUME = 0,
    BIT_SOLID,
    // その他、暗号化／リカバリレコード／信頼証明／自己展開／ロックなどなど
    BIT_MAX,
  };

  szstring archiveComment;
  u64 compressedSize;
  u64 uncompressedSize;
  u32 numberOfItems;
  u32 numberOfVolumes;
  std::bitset<BIT_MAX> bits;

};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_ARCHIVE_PROPERTY)
